<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Widget_Base;
use Elementor\Control_Media;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Background;
use Elementor\Repeater;


class MOElement_Heading_fancy extends MOElement_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'mo-heading-fancy';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Heading fancy', 'saga' );
	}
	/**
	 * Get widget icon.
	 */
	public function get_icon() {
		return 'eicon-t-letter mo-element';
	}
	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'heading', 'title', 'text' ];
	}
	public function get_script_depends() {
		return ['split'];
	}
	public function get_style_depends() {
		return array();
	}
	
	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			array(
				'label' => __( 'Title', 'saga' ),
			)
		);
		
		$this->add_control(
			'content',
			array(
				'label' => __( 'Title', 'saga' ),
				'type' => Controls_Manager::TEXTAREA,
				'dynamic' => array(
					'active' => true,
				),
				'rows' => '3',
				'default' => 'Add Your Heading Text Here',
				'description' => esc_html__( 'you can add Highlight between <code></code>', 'saga' ),
			)
		);

		

		$this->add_responsive_control(
			'alignment',
			[
				'label' => __( 'Alignment', 'saga' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'saga' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'saga' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'saga' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .mo-heading-fancy .mo-element, {{WRAPPER}} .mo-heading-fancy' => 'text-align: {{VALUE}}',
				],
				'toggle' => true,
			]
		);
		$this->add_control(
			'link',
			[
				'label' => __( 'URL (Link)', 'saga' ),
				'type' => Controls_Manager::URL,
				'placeholder' => __( 'https://your-link.com', 'saga' ),
				'show_external' => true,
				'default' => [
					'url' => '',
					'is_external' => false,
					'nofollow' => false,
				],
				'separator' => 'before',
			]
		);
		$this->end_controls_section();


		// Animation Section
		$this->start_controls_section(
			'animation_section',
			[
				'label' => __( 'Effects', 'saga' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'highlight_type',
			[
				'label' => __( 'Highlight Type', 'saga' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'mo-highlight-underline',
				'options' => [
					''                    => __( 'None', 'saga' ),
					'mo-highlight-style1' => __( 'Style 1', 'saga' ),
					'mo-highlight-style2' => __( 'Style 2', 'saga' ),
					'mo-highlight-style3' => __( 'Style 3', 'saga' ),
					'mo-highlight-style4' => __( 'Style 4', 'saga' ),
					'mo-highlight-style5' => __( 'Style 5', 'saga' ),
				],
			]
		);
		$this->add_control(
			'highlight_color',
			array(
				'label' => __( 'Highlight Color', 'saga' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .mo-heading-fancy svg path' => 'fill:{{VALUE}}; stroke:{{VALUE}};',
				],
				'condition' => array(
					'highlight_type' => [ 'mo-highlight-style1', 'mo-highlight-style2', 'mo-highlight-style3', 'mo-highlight-style4', 'mo-highlight-style5' ],
				),
			)
		);
		$this->add_control(
			'enable_split',
			[
				'label' => __( 'Enable text split?', 'saga' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'On', 'saga' ),
				'label_off' => __( 'Off', 'saga' ),
				'return_value' => 'true',
				'default' => '',
				'separator' => 'before',
				'frontend_available' => true
			]
		);

		$this->add_control(
			'split_type',
			[
				'label' => __( 'Splitting type', 'saga' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'lines',
				'options' => [
					'lines'  => __( 'Lines', 'saga' ),
					'words' => __( 'Words', 'saga' ),
					'chars, words' => __( 'Characters', 'saga' ),
				],
				'condition' => [
					'enable_split' => 'true',
				],
				'frontend_available' => true
			]
		);

		$this->add_control(
			'enable_txt_rotator',
			[
				'label' => __( 'Enable text rotator?', 'saga' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'On', 'saga' ),
				'label_off' => __( 'Off', 'saga' ),
				'return_value' => 'yes',
				'default' => 'no',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'rotator_type',
			[
				'label' => __( 'Animation type', 'saga' ),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => [
					''  => __( 'Slide', 'saga' ),
					'basic' => __( 'Basic', 'saga' ),
				],
				'condition' => array(
					'enable_txt_rotator' => 'yes',
				),
			]
		);

		$this->add_control(
			'rotator_delay', [
				'label' => __( 'Words stay time', 'saga' ),
				'description' => __( 'Stay time for each word in seconds. Default is 2 seconds', 'saga' ),
				'type' => Controls_Manager::TEXT,
				'condition' => array(
					'enable_txt_rotator' => 'yes',
				),
			]
		);

		$this->add_control(
			'word_colors',
			[
				'label' => __( 'Words color', 'saga' ),
				'type' => Controls_Manager::COLOR,
				'condition' => array(
					'enable_txt_rotator' => 'yes',
				),
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} .txt-rotate-keywords' => 'color: {{VALUE}}',
				],
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'word', [
				'label' => __( 'Title word', 'saga' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'word_color',
			[
				'label' => __( 'Color', 'saga' ),
				'type' => Controls_Manager::COLOR,
			]
		);

		$this->add_control(
			'items',
			[
				'label' => __( 'Text rotator words', 'saga' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ word }}}',
				'condition' => array(
					'enable_txt_rotator' => 'yes',
				),
			]
		);
		$this->end_controls_section();

		
		// Style Tab
		$this->start_controls_section(
			'style_section',
			[
				'label' => __( 'Style', 'saga' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
        $this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'content_typography',
				'label' => __( 'Typography', 'saga' ),
				'selector' => '{{WRAPPER}} .mo-heading-fancy .mo-element',
			]
		);
		$this->add_control(
			'color',
			array(
				'label' => __( 'Color', 'saga' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .mo-heading-fancy h3, {{WRAPPER}} .mo-heading-fancy .mo-element' => 'color:{{VALUE}}',
				],
				'separator' => 'before',
			)
		);
		$this->add_control(
			'add_gradient',
			[
				'label' => __( 'Gradient Color', 'saga' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'On', 'saga' ),
				'label_off' => __( 'Off', 'saga' ),
				'return_value' => 'yes',
				'default' => ''	
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'color2',
				'label'    => __( 'Fancy Heading Gradient', 'saga' ),
				'types'    => [  'gradient' ],	
				'selector' => '{{WRAPPER}} .mo-heading-fancy .mo-gradient-heading, {{WRAPPER}} .mo-heading-fancy .mo-gradient-heading .split-unit, {{WRAPPER}} .mo-heading-fancy .mo-gradient-heading .mo-highlight-txt',
				'condition' => array(
					'add_gradient' => 'yes',
				)
			]
		);
		
		$this->end_controls_section();
		$this->end_controls_tab();
	}

	protected function add_render_attributes() {
		parent::add_render_attributes();
		$settings = $this->get_settings();
		$classnames = [];
		if ( $settings['enable_split'] !== '' ) {
			array_push($classnames, 'mo-el-has-inner-anim');
		}
		$this->add_render_attribute( '_wrapper', 'class', $classnames );
	}

	protected function content_template() {

		?>
		<#

		const wrapperClassnames = [
			settings.hover_text_outline,
			settings.outline_appearance,
		].filter(classname => classname !== '');

	
		const titleClassnames = [
			settings.add_gradient === 'yes' && settings.color2 !== '' ? 'mo-gradient-heading' : '',
			settings.whitespace,
			settings.vertical_txt === 'true' ? 'text-vertical' : '',
			settings.enable_split === 'true' ? getSplitClassnames() : '',
		
		].filter(classname => classname !== '');

		const title = settings.content;
		const link = settings.link.url;

		const wrapperAttrs = {
			'class': ['mo-heading-fancy', wrapperClassnames.join(' ')],
			'id': view.model.get('id')
		};

		const titleAttrs = {
			'class': [ 'mo-element', titleClassnames.join(' '), 'elementor-inline-editing' ],
			'data-elementor-setting-key': 'content',
			'data-elementor-inline-editing-toolbar': 'basic'
		};

		function getSplitClassnames() {
			const {split_type} = settings;
			let classname = 'mo-split-';

			if ( split_type === 'chars, words' ) {
				classname += `${classname}chars`;
			} else {
				classname += `${classname}${split_type}`;
			}
			return classname;
		}


		function getTextRotatorWords() {

			if (settings.enable_txt_rotator !== 'yes'){
				return '';
			}

			const words = settings.items;
			let out = '';
			let style_word = '';

			out += ' <span class="txt-rotate-keywords">';
			if ( settings.rotator_type === 'list' ) {
				out += ' <span class="txt-rotate-keywords-inner d-block h-100">';
			}
			if ( words ) {
				_.each( words, function( word, i ) {
					var active = ( i === 0 ) ? ' active' : '';
					style_word = ( word.word_color ) ? 'style="color:' + word.word_color + '"' : '';					
					out += '<span class="txt-rotate-keyword" ' + style_word + '><span>' + word.word + '</span></span>';
				} );
			}

			if ( settings.rotator_type === 'list' ) {
				out += ' </span>';
			}
			out += '</span>';
			return out;
		}



		const getRotatorOpts = (() => {

			if( settings.enable_txt_rotator !== 'yes' ) {
				return '';
			}

			titleAttrs['data-text-rotator'] = true;
			const options = {};

			if( settings.rotator_type === 'basic' ) {
				//options['animationType'] = "basic";
			}
			if ( settings.rotator_delay !== '' ) {
				options['delay'] = settings.rotator_delay;
			}

			if ( options ) {
				titleAttrs['data-text-rotator-options'] = JSON.stringify( options );
			}

		})();

	
		
		// Split Animation
		const getSplitOptions = (() => {

			if( settings.enable_split === '' ) {
				return;
			}

			titleAttrs['data-split-text'] = true;
			titleAttrs['data-split-options'] = JSON.stringify( { "type": settings.split_type } );
			
		})();

		view.addRenderAttribute( 'wrapperAttributes', wrapperAttrs);

		view.addRenderAttribute( 'titleAttributes', titleAttrs);

		let {content} = settings;

		content = content.replace('/<p>/g', '');
		content = content.replace('/</p>/g', '');

		#>
			<div {{{ view.getRenderAttributeString('wrapperAttributes') }}}>
				<h3{{{ view.getRenderAttributeString('titleAttributes') }}} >
					<# if( settings.hover_text_outline === 'mo-fh-outline' ) { #>
						<span class="mo-fh-txt-outline">{{{content}}}</span>
					<# } #>
					
						{{{ content + getTextRotatorWords() }}}
				
					</h3>
			</div>
		<?php

	}

	protected function fancy_title() {

		$settings = $this->get_settings_for_display();
		
		
		$classnames = $outline_title = '';
		$classnames_arr = array('mo-element');
	
			array_push($classnames_arr, 'elementor-heading-title');
		
		
		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			$classnames_arr[] = 'elementor-inline-editing';
			$inline_edit_attr = 'data-elementor-setting-key="content" data-elementor-inline-editing-toolbar="basic"';
		} else {
			$inline_edit_attr = '';
		}
		
		$link = isset($settings['link']['url']) ? $settings['link']['url'] : '';
		
		if( !empty( $settings['add_gradient'] ) ) {
			$classnames_arr[] = 'mo-gradient-heading';
		}
		
		if( !empty( $settings['whitespace'] ) ) {
			$classnames_arr[] = $settings['whitespace'];
		}
		
		if( !empty( $settings['highlight_type'] ) ) {
			$highlight_type = $settings['highlight_type'];
			$highlight_classname = $highlight_type;
			if ( $highlight_type === 'mo-highlight-underline' ) {
				$highlight_classname = 'mo-highlight-classic';
			} else if ( $highlight_type === 'mo-highlight-style1' ) {
				$highlight_classname = 'mo-highlight-style mo-highlight-style1';
			} else if ( $highlight_type === 'mo-highlight-style-style2 mo-highlight-style2' ) {
				$highlight_classname = 'mo-highlight-style mo-highlight-style2';
			}
		
			$classnames_arr[] = $highlight_classname;
		}
		if ( $settings['enable_split'] ) {
			$classnames_arr[] = $this->get_split_classname();
		}

		$content = $settings['content'];
		$content = preg_replace('/^<p[^>]*>(.*)<\/p[^>]*>/', '$1', $content);
		$title   = $settings['content'];
		$title   = preg_replace('/^<p[^>]*>(.*)<\/p[^>]*>/', '$1', $title);
		$title   = $title . $this->get_title_words() ; 
		$code    = "<code>";
		$endcode = "</code>";
		$svg = $this->getSVGHighlight();
		$SVGhighlight = '<mark class="mo-highlight"><span class="mo-highlight-txt">' . $svg;
	    
		if (strpos($content, $code) !== false) {
			$content1 = array("<code>", "</code>");
			$content2 = array($SVGhighlight,'</span></mark>');
			$highlight = str_replace($content1, $content2 , $title);
			$title = $highlight ; 
		} else { 
			$title = do_shortcode( wp_kses_post( $title ) ); 
		} 

		if( !empty( $classnames_arr ) ) {
			$classnames = 'class="' . join( ' ', $classnames_arr ) . '"';
		}
		
		// Title
		if( $title ) {
			if ( !empty ( $link ) ) {
				$link_target = $settings['link']['is_external'] === 'on' ? '_blank' : '_self';
				$link = " href=$link " . " target=$link_target ";
				printf( '<%1$s %3$s %4$s><a%5$s %7$s'. $this->get_split_options() .'>%6$s %2$s</a></%1$s>', 'h3',  $title , $classnames, $this->get_data_opts(), esc_attr( $link ), $outline_title, $inline_edit_attr);

			}
			else {
				printf( '<%1$s %3$s %4$s %5$s'. $this->get_split_options() .'> %2$s </%1$s>', 'h3',
				 $title , $classnames, $this->get_data_opts(), $inline_edit_attr);
			}
			
		}
		}
		

	
		protected function getSVGHighlight() {
		$settings = $this->get_settings_for_display();
		 $svg = '';
		if ( $settings['highlight_type'] == 'mo-highlight-style1' ) {
			$svg = '<svg class="mo-highlight-style1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 500 150" preserveAspectRatio="none"><path d="M15.2,133.3L15.2,133.3c121.9-7.6,244-9.9,366.1-6.8c34.6,0.9,69.1,2.3,103.7,4"></path></svg>';
		} else if ( $settings['highlight_type'] == 'mo-highlight-style2' ) {
			$svg = '<svg class="mo-highlight-style2" width="233" height="13" viewBox="0 0 233 13" xmlns="http://www.w3.org/2000/svg" aria-hidden="true" preserveAspectRatio="none"><path d="m.624 9.414-.312-2.48C0 4.454.001 4.454.002 4.454l.035-.005.102-.013.398-.047c.351-.042.872-.102 1.557-.179 1.37-.152 3.401-.368 6.05-.622C13.44 3.081 21.212 2.42 31.13 1.804 50.966.572 79.394-.48 113.797.24c34.387.717 63.927 2.663 84.874 4.429a1048.61 1048.61 0 0 1 24.513 2.34 641.605 641.605 0 0 1 8.243.944l.432.054.149.02-.318 2.479-.319 2.48-.137-.018c-.094-.012-.234-.03-.421-.052a634.593 634.593 0 0 0-8.167-.936 1043.26 1043.26 0 0 0-24.395-2.329c-20.864-1.76-50.296-3.697-84.558-4.413-34.246-.714-62.535.332-82.253 1.556-9.859.612-17.574 1.269-22.82 1.772-2.622.251-4.627.464-5.973.614a213.493 213.493 0 0 0-1.901.22l-.094.01-.028.004Z"/></svg>';
		} else if ( $settings['highlight_type'] == 'mo-highlight-style3' ) {
			$svg = '<svg class="mo-highlight-style3" viewBox="0 0 491 26" fill="none" xmlns="http://www.w3.org/2000/svg"><path class="line1" fill-rule="evenodd" clip-rule="evenodd" d="M208.549 3.29849C185.59 4.08616 162.901 5.05468 140.776 6.11351C109.455 7.61138 78.1462 9.16091 47.0261 11.6143C37.2302 12.3891 27.2823 12.8797 17.5476 13.8482C11.4368 14.4551 3.334 15.3201 2.27061 15.5009C1.71441 15.6042 1.47176 15.7593 1.39336 15.8109C0.842061 16.1725 0.942652 16.521 1.25383 16.7921C1.37879 16.9084 1.69478 17.1925 2.57195 17.2313C61.2348 19.8784 121.221 14.7133 179.955 14.171C281.81 13.2413 386.752 16.9343 487.799 24.9918C488.681 25.0564 489.539 24.7334 489.661 24.2557C489.808 23.7908 489.171 23.3389 488.289 23.2743C387.071 15.2039 281.957 11.498 179.906 12.4406C125.168 12.9442 69.3499 17.4766 14.5363 15.9271C15.7957 15.798 17.0307 15.6688 18.1578 15.5526C27.8533 14.5841 37.7594 14.1064 47.5162 13.3316C78.5701 10.8782 109.815 9.32872 141.095 7.84376C179.906 5.98434 220.408 4.38316 261.228 3.63423C275.831 3.75044 290.386 3.86671 304.94 4.00875C336.425 4.31865 368.057 5.22254 399.469 6.43633C408.927 6.8108 418.385 7.19814 427.842 7.53387C430.979 7.65009 439.064 7.98576 440.191 7.95993C441.588 7.93411 441.857 7.31436 441.882 7.21106C441.955 6.97863 441.906 6.65579 441.22 6.38463C441.147 6.34589 440.706 6.21668 439.726 6.13921C382.611 1.56813 321.626 0.806405 261.277 1.90398C197.621 1.42621 133.719 1.21951 70.2247 1C69.3157 1 68.5733 1.38744 68.5659 1.8652C68.561 2.34297 69.2935 2.73042 70.2026 2.74333C116.178 2.89828 162.387 3.05315 208.549 3.29849Z" fill="none" stroke="none"></path></svg>';
		} else if ( $settings['highlight_type'] == 'mo-highlight-style4' ) {
			$svg = '<svg class="mo-highlight-style4" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 269.06 7"><path d="M213.63,1a7.93,7.93,0,0,0,1.76.17h1.32a8.05,8.05,0,0,0,2.2-.13l5.27.13,8.79,0,1.32.17c1.32,0,2.2,0,3.08,0s1.76.09,2.64.09c0,0,1.32-.13,3.52-.21l3.08,0c1.76,0,3.08,0,4,.09,1.32,0,2.2,0,3.08,0,2.64.21,4.39.3,6.15.3l-.44.17-2.2,0v.17H258c.88,0,2.2,0,3.52,0C270.31,2,269,4.32,269,4.32s-5.71.51-11.42.6c.88.13,1.32.21.88.26-1.32,0-2.64,0-4,0l-.44.17c2.64,0-13.62.47-15.38.72-4.39,0-8.79.09-12.74.26l-5.27-.13c0,.13-5.27.21-15.82.21l-19.77.21-1.76-.09c-23.29,0-52.29.09-86.56.09,0,0-2.64,0-4,0-18.45,0-30.32,0-35.15,0l-7-.09a4.45,4.45,0,0,1-1.32-.09c-2.64,0-11,0-24.61,0l-4-.13-.44.17-3.08,0c-12.3-.38-18-1-17.14-1.66a4.63,4.63,0,0,0,1.32-.21L1,4.11H2.28C1.41,3.85,1,3.64,1,3.56a28.61,28.61,0,0,0,3.52-.21C4,3,4,2.87,4,2.75H5.36c-.88-.09-.88-.17-.88-.21l1.32-.3s0-.09-.88-.13c2.2-.47,3.52-.85,3.52-1A5.35,5.35,0,0,1,6.68.92,15.24,15.24,0,0,1,10.19.79h3.08s4-.13,11.86-.17A6,6,0,0,1,26.89.45l1.32,0c1.32,0,1.76,0,1.76.13L52.37.75c3.08,0,4.83-.13,4.83-.17-.88,0-.88,0-.88-.13l1.76,0V.58S77.86.62,82.25.45l3.08,0a7.62,7.62,0,0,1,2.2-.17h.88l5.71.47c6.59,0,18.89,0,18.89,0a7.09,7.09,0,0,1,1.76.17L124,.79c2.2,0,4.39,0,7,.13h18l-.44,0c1.32,0,2.64,0,3.52,0,20.21.09,30.76.09,31.2,0l4.83.09q2,.06,4,0C193,1,193,1,193,1c10.11,0,16.7,0,20.65,0Z"></path></svg>';
		} else if ( $settings['highlight_type'] == 'mo-highlight-style5' ) {
			$svg = '<svg class="mo-highlight-style5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 500 150" preserveAspectRatio="none"><path d="M15,133.4c19-12.7,48.1-11.4,69.2-8.2 c6.3,1.1,12.9,2.1,19.2,3.4c16.5,3.2,33.5,6.3,50.6,5.5c12.7-0.6,24.9-3.4,36.7-6.1c11-2.5,22.4-5.1,34.2-5.9 c24.3-1.9,48.5,3.4,71.9,8.4c27.6,6.1,53.8,11.8,80.4,6.8c9.9-1.9,19.2-5.3,28.3-8.4c8.2-3,16.9-5.9,25.9-8 c20.3-4.4,45.8-1.1,53.6,12.2" style="animation-play-state: running;"></path></svg>';
		} 
		return $svg;
	}
	
	protected function get_text_rotator_options() {

		$settings = $this->get_settings_for_display();
		
		if( $settings['enable_txt_rotator'] !== 'yes' ) {
			return;
		}
		$attrs = array(
			'data-text-rotator' => true,
		);
		$options = array();

		if( 'basic' === $settings['rotator_type'] ) {
			//$options['animationType'] = 'basic';
		}

		if ( ! empty( $settings['rotator_delay'] ) ) {
			$options['delay'] = (float) $settings['rotator_delay'];
		}

		if ( ! empty( $options ) ) {
			$attrs['data-text-rotator-options'] = wp_json_encode( $options );
		}
		return $attrs;
	}

	protected function get_data_opts() {
		$settings = $this->get_settings_for_display();
		$opts = array();
		//$rotator_opts = $this->get_text_rotator_options();
		$highlight_opts = $this->get_highlight_opts();

		//if( is_array( $rotator_opts ) && ! empty( $rotator_opts ) ) {
		//	$opts = array_merge( $opts, $rotator_opts );
		//}
		if( is_array( $highlight_opts ) && ! empty( $highlight_opts ) ) {
			$opts = array_merge( $opts, $highlight_opts );
		}
		//return $this->html_attributes( $opts );  
		foreach($opts as $opt){
			echo $opt . "\n";
		}
	}

	protected function get_highlight_opts() {
		$settings = $this->get_settings_for_display();
		if( !has_shortcode( $settings['content'], 'mo-highlight' )  ) {
			return;
		}
		$opts = array(
			'data-inview' => true,
			'data-transition-delay' => true,
			'data-delay-options' => wp_json_encode(
				array(
					'elements' => '.mo-highlight-inner',
					'delayType' => 'transition'
				)
			)
		);
		return $opts;
	}

	protected function get_split_classname() {

		$settings = $this->get_settings_for_display();

		$classname = '';
		$prefix = 'mo-split-';
		$split_type = $settings['split_type'];

		if ( ! $settings['enable_split'] ) {
			return $classname;
		};

		if ( $split_type === 'chars, words' ) {
			$classname = $prefix . 'chars';
		} else {
			$classname = $prefix . $split_type;
		}
		return $classname;
	}

	protected function get_title_words() {

		$settings = $this->get_settings_for_display();
		
		if( $settings['enable_txt_rotator'] !== 'yes' ) {
			return;
		}

		if( empty( $settings['items'] ) ) {
			return;
		}

		$words = $settings['items'] ;

		if( empty( $words ) ) {
			return;
		}
		
		$out = $style_word = '';
		
		$out .= ' <span class="txt-rotate-keywords">';
		if ( $settings['rotator_type'] === 'list' ) {
			$out .= ' <span class="txt-rotate-keywords-inner d-block h-100">';
		}
		$i = 1;
		foreach ( $words as $word ) {
			$active = ( $i == 1 ) ? ' active' : '';
			$style_word = !empty( $word['word_color'] ) ? 'style="color:' . esc_attr( $word['word_color'] ) . '"' : '';
			
			$out .= '<span class="txt-rotate-keyword' . $active . '" ' . $style_word . '><span>' . esc_html( $word['word'] ) . '</span></span>';
			$i++;
		}
		if ( $settings['rotator_type'] === 'list' ) {
			$out .= ' </span>';
		}
		$out .= '</span>';

		return $out;
		
	}

	protected function get_split_options() {

		extract( $this->get_settings_for_display() );

		if( !$enable_split ) {
			return;
		}
	
		$opts = $split_opts = array();
		$split_opts['type'] = $split_type;

		$ca_opts['triggerHandler'] = 'inview';
		$ca_opts['animationTarget'] = '.split-inner';
		$ca_opts['duration'] = '1200';
		$ca_opts['startDelay'] = '35';
		$ca_opts['delay'] = '120';
		$ca_opts['easing'] ='easeInQuad';
		$ca_opts['direction'] = 'forward';
	    $ca_opts['initValues'] =  array( 'translateY' => 16 ,'opacity' => 0 );
		$ca_opts['animations'] =  array( 'translateY' => 0 ,'opacity' => 1 );

	    $opts[] = 'data-ca-options=\'' . wp_json_encode( $ca_opts ) . '\'';
		$opts[] = 'data-split-text="true"';
		$opts[] = 'data-custom-animations="true"';
		$opts[] = 'data-fittext="true"';
	    $opts[] = 'data-text-rotator="true"';
		return join( ' ', $opts );

	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		printf( '<div class="mo-element-%s mo-element">', $this->get_name() );
		 include $this->get_template('mo-heading-fancy.php');
	  print '</div>';
	}
}